# -*- coding: utf-8 -*-
from ctypes import *
from enum import Enum
dll = cdll.LoadLibrary("./PlayerOnePW.dll")


#************************Type Definition************************
# Player One Phoenix Filter Wheel referred to as PW

class PWErrors(Enum):                      
    '''Return Error Code Definition'''
    PW_OK = 0                           # operation successful
    PW_ERROR_INVALID_INDEX = 1          # invalid index, means the index is < 0 or >= the count
    PW_ERROR_INVALID_HANDLE = 2         # invalid PW handle
    PW_ERROR_INVALID_ARGU = 3           # invalid argument(parameter)
    PW_ERROR_NOT_OPENED = 4             # PW not opened
    PW_ERROR_NOT_FOUND = 5              # PW not found, may be removed
    PW_ERROR_IS_MOVING = 6              # PW is moving
    PW_ERROR_POINTER = 7                # invalid pointer, when get some value, do not pass the NULL pointer to the function
    PW_ERROR_OPERATION_FAILED = 8       # operation failed (Usually, this is caused by sending commands too often or removed)
    PW_ERROR_FIRMWARE_ERROR = 9         # firmware error (If you encounter this error, try calling POAResetPW)

class PWState(Enum):         
    '''PW State Definition'''
    PW_STATE_CLOSED = 0                 # PW was closed
    PW_STATE_OPENED = 1                 # PW was opened, but not moving(Idle)
    PW_STATE_MOVING = 2                 # PW is moving

class PWProperties(Structure):               
    '''PW Properties Definition'''
    _fields_ = [("Name",c_char*64),         # the PW name
                ("Handle",c_int),           # it's unique,PW can be controlled and set by the handle
                ("PositionCount",c_int),    # the filter capacity, eg: 5-position
                ("SN",c_char*32),           # the serial number of PW,it's unique
                ("reserved",c_char*32)]     # reserved

MAX_NAME_LEN = 24 # the max length of custom name and filter alias

#************************Function Definition************************
def GetPWCount():
    """
    get connected filter wheel count

    Returns:
        int: the counts of POA filter wheels connected to the computer host
    """
    dll.POAGetPWCount.restype = c_int
    return dll.POAGetPWCount()

def GetPWProperties(nIndex):
    """
    get the property of the connected filter wheel, NO need to open the filter wheel for this operation

    Args:
        nIndex (int): the range: [0, count), note: index is not Handle

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
        PWProperties: property of filter wheel
    """
    PWProp = PWProperties()
    Func = dll.POAGetPWProperties
    Func.argtypes = [c_int,POINTER(PWProperties)]
    Func.restype = PWErrors
    Status = Func(nIndex,byref(PWProp))
    return Status, PWProp

def GetPWPropertiesByHandle(nHandle):
    """
    get the property of the connected filter wheel by Handle, it's a convenience function to get the property of the known Handle of filter wheel

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
        PWProperties: property of filter wheel
    """
    PWProp = PWProperties()
    Func = dll.POAGetPWPropertiesByHandle
    Func.argtypes = [c_int,POINTER(PWProperties)]
    Func.restype = PWErrors
    Status = Func(nHandle,byref(PWProp))
    return Status, PWProp

def OpenPW(nHandle):
    """
    open the filter wheel, note: the following API functions need to open the filter wheel first

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
    """
    dll.POAOpenPW.restype = PWErrors
    return dll.POAOpenPW(nHandle)

def ClosePW(nHandle):
    """
    close the filter wheel

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
    """
    dll.POAClosePW.restype = PWErrors
    return dll.POAClosePW(nHandle)

def GetCurrentPosition(nHandle):
    """
    get the Current Position of filter wheel

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
        int: current position of filter wheel
    """
    dll.POAGetCurrentPosition.restype = PWErrors
    currPos = c_int(0)
    Status = dll.POAGetCurrentPosition(nHandle, byref(currPos))    
    return Status, currPos.value
    

def GotoPosition(nHandle,nPosition):
    """
    let the filter wheel goto a filter position

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties
        nPosition (int): the filter position, range:[0, PositionCount-1]

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
    """
    dll.POAGotoPosition.restype = PWErrors
    Status = dll.POAGotoPosition(nHandle, nPosition)
    return Status

def GetOneWay(nHandle):
    """
    get the rotation of filter wheel is unidirectional

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
        int: current position of filter wheel
    """
    dll.POAGetOneWay.restype = PWErrors
    isOneWay = c_int(0)
    Status = dll.POAGetOneWay(nHandle, byref(isOneWay))    
    return Status, bool(isOneWay.value)

def SetOneWay(nHandle,isOneWay):
    """
    set the rotation of filter wheel to unidirectional or not

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties
        isOneWay (bool): True means unidirectional, False means not unidirectional

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
    """
    dll.POASetOneWay.restype = PWErrors
    Status = dll.POASetOneWay(nHandle, int(isOneWay))
    return Status

def GetPWState(nHandle):
    """
    get the current state of filter wheel, this is for easy access to whether filter wheel is moving

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
        PWState: current state of filter wheel
    """
    dll.POAGetPWState.restype = PWErrors
    PW_State=c_int(0)
    Status = dll.POAGetPWState(nHandle,byref(PW_State))
    return Status, PWState(PW_State.value)

def GetPWFilterAlias(nHandle,nPosition):
    """
    get the filter alias from wheel(eg: R, G, or IR-cut)

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties
        nPosition (int): the filter position, range:[0, PositionCount-1]

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
        str: filter alias for this position
    """
    dll.POAGetPWFilterAlias.argtypes = [c_int, c_int, c_char_p, c_int]
    dll.POAGetPWFilterAlias.restype = PWErrors
    buf = create_string_buffer(MAX_NAME_LEN)
    Status = dll.POAGetPWFilterAlias(nHandle, nPosition, buf, MAX_NAME_LEN)
    return Status, buf.value.decode('utf-8')

def SetPWFilterAlias(nHandle,nPosition,strAlias):
    """
    set the filter alias(eg: you can set alias of 1 position to R)

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties
        nPosition (int): the filter position, range:[0, PositionCount-1]
        strAlias (str): filter alias for this position, max length is MAX_NAME_LEN, NOTE: If it's empty, the previous value will be cleared

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
    """
    dll.POASetPWFilterAlias.argtypes = [c_int, c_int, c_char_p, c_int]
    dll.POASetPWFilterAlias.restype = PWErrors
    buf = strAlias.encode('utf-8')
    bufLen = len(strAlias)
    if bufLen > MAX_NAME_LEN:
        bufLen = MAX_NAME_LEN
    return dll.POASetPWFilterAlias(nHandle, nPosition, buf, bufLen)
    
def GetPWFocusOffset(nHandle,nPosition):
    """
    get focus offset for this filter(each filter has different focus offset, using this with Auto Focuser)

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties
        nPosition (int): the filter position, range:[0, PositionCount-1]

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
        int: focus offset for this filter
    """
    dll.POAGetPWFocusOffset.argtypes = [c_int, c_int, POINTER(c_int)]
    dll.POAGetPWFocusOffset.restype = PWErrors
    focusOffset = c_int(0)
    Status = dll.POAGetPWFocusOffset(nHandle, nPosition, byref(focusOffset))
    return Status, focusOffset.value

def SetPWFocusOffset(nHandle,nPosition,nFocusOffset):
    """
    set focus offset for this filter

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties
        nPosition (int): the filter position, range:[0, PositionCount-1]
        nFocusOffset (int): focus offset for this filter

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
    """
    dll.POASetPWFocusOffset.argtypes = [c_int, c_int, c_int]
    dll.POASetPWFocusOffset.restype = PWErrors
    return dll.POASetPWFocusOffset(nHandle, nPosition, nFocusOffset)
    
def GetPWCustomName(nHandle):
    """
    get the user custom name for wheel(eg: MyPW, and the PW Name will show as PhoenixWheel[MyPW])

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
        str: custom name for wheel
    """
    dll.POAGetPWCustomName.argtypes = [c_int, c_char_p, c_int]
    dll.POAGetPWCustomName.restype = PWErrors
    buf = create_string_buffer(MAX_NAME_LEN)
    Status = dll.POAGetPWCustomName(nHandle, buf, MAX_NAME_LEN)
    return Status, buf.value.decode('utf-8')

def SetPWCustomName(nHandle,strCustomName):
    """
    set the user custom name for wheel

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties
        strCustomName (str): custom name for wheel, max length is MAX_NAME_LEN, NOTE: If it's empty, the previous value will be cleared

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
    """
    dll.POASetPWCustomName.argtypes = [c_int, c_char_p, c_int]
    dll.POASetPWCustomName.restype = PWErrors
    buf = strCustomName.encode('utf-8')
    bufLen = len(strCustomName)
    if bufLen > MAX_NAME_LEN:
        bufLen = MAX_NAME_LEN
    return dll.POASetPWCustomName(nHandle, buf, bufLen)

def ResetPW(nHandle):
    """
    reset the wheel, If PW_ERROR_FIRMWARE_ERROR occurs, call this function

    Args:
        nHandle (int): Handle of filter wheel, get it from in the PWProperties

    Returns:
        PWErrors: error code returned by calling this function, PW_OK indicates success
    """
    dll.POAResetPW.restype = PWErrors
    return dll.POAResetPW(nHandle)

def GetPWErrorString(err):
    """
    convert PWErrors enum to string, it is convenient to print or display errors

    Args:
        err (PWErrors): a error returned by the API function

    Returns:
        str: string error
    """
    dll.POAGetPWErrorString.argtypes = [c_int]
    dll.POAGetPWErrorString.restype = c_char_p
    buf = dll.POAGetPWErrorString(err.value)
    return buf.decode('utf-8')

def GetPWAPIVer():
    """
    get the API version

    Returns:
        int: API version, easy to do version comparison, eg: 20240420
    """
    dll.POAGetPWAPIVer.restype = c_int
    return dll.POAGetPWAPIVer()

def GetPWSDKVer():
    """
    get this sdk version

    Returns:
        str: sdk version string(major.minor.patch), eg: "1.0.1"
    """
    dll.POAGetPWSDKVer.restype = c_char_p
    buf = dll.POAGetPWSDKVer()
    return buf.decode('utf-8')